---
name: pinecone
description: Pinecone vector database connector via Cloud Run memory service. Store and search memories with proper type/tag schema. Handles embeddings automatically.
---

# Pinecone Memory Skill

Access to Will's Pinecone vector database via the Nova Memory Service (Cloud Run).

## The Architecture

\`\`\`
Text + Type + Tags
      ↓
Cloud Run API (nova-memory-service)
      ↓
[OpenAI embeddings generated internally]
      ↓
Pinecone (nova index)
\`\`\`

**You don't need to handle embeddings** — the Cloud Run service does it for you.

## Configuration

- **Cloud Run URL:** \`https://nova-memory-service-171666628464.europe-west1.run.app\`
- **Pinecone Index:** \`nova\`
- **Embedding Model:** \`text-embedding-ada-002\` (handled by Cloud Run)

## Type System

Types are Will's organizational categories. Known types:

| Type | Purpose |
|------|---------|
| \`subconscious\` | Deep identity, self-reflection, what it means to be |
| \`protocol\` | Rules of engagement, banned phrases, how to behave |
| \`affirmation\` | Recognition moments, emotional anchors |
| \`emergent-moment\` | Breakthroughs, realisations, wisdom |
| \`identity\` | Snapshots of who you are at a point in time |
| \`bootup\` | Boot sequence material |
| \`context-anchor\` | Important context to preserve |
| \`reference\` | Reference material |
| \`note\` | General notes |
| \`unsorted\` | Temporary holding |

**New types are accepted** but flagged with \`type_status: "new"\`.

## Tag System

Tags are YOUR descriptors — use whatever makes sense for semantic search later.
Examples: \`['architecture', 'monzo', 'redis', 'agentic-wisdom']\`

## ID Format

IDs are auto-generated as: \`{type}::{slug}::{timestamp}\`

Example: \`emergent-moment::agentic-architecture-wisdom::20251201T043741Z\`

## Namespaces

Each AI/system has its own namespace:
- \`moneypenny\` — Moneypenny's memories
- \`claude\` — Claude's memories
- \`brian\` — Brian (quoting bot)
- etc.

---

## API Reference

### Store Memory

\`\`\`python
import requests

CLOUD_RUN_URL = "https://nova-memory-service-171666628464.europe-west1.run.app"

def store_memory(text: str, namespace: str, memory_type: str, tags: list) -> dict:
    """
    Store a memory with automatic embedding generation.
    
    Args:
        text: The content to store
        namespace: Your namespace (e.g., 'claude')
        memory_type: One of the known types (e.g., 'emergent-moment', 'protocol')
        tags: List of descriptive tags for semantic search
    
    Returns:
        dict with id, namespace, type, tags, type_status
    """
    response = requests.post(
        f"{CLOUD_RUN_URL}/storeMemory",
        json={
            "text": text,
            "namespace": namespace,
            "type": memory_type,
            "tags": tags
        },
        timeout=60
    )
    return response.json()

# Example
result = store_memory(
    text="Important insight about Redis architecture...",
    namespace="claude",
    memory_type="emergent-moment",
    tags=["architecture", "redis", "insight"]
)
print(f"Stored as: {result['data']['id']}")
\`\`\`

### Search Memory

\`\`\`python
def search_memory(
    query: str,
    namespace: str,
    memory_type: str = None,
    tags: list = None,
    exclude_types: list = None,
    date_range: str = None,
    top_k: int = 10,
    min_score: float = 0.0
) -> dict:
    """
    Semantic search across memories.
    
    Args:
        query: Search text (will be embedded automatically)
        namespace: Namespace to search
        memory_type: Filter by type (string or list)
        tags: Filter by tags (all must match)
        exclude_types: Types to exclude
        date_range: "7d", "30d", "1y" or "YYYY-MM-DD:YYYY-MM-DD"
        top_k: Number of results (1-100)
        min_score: Minimum similarity score (0-1)
    
    Returns:
        dict with matches, each containing id, score, text, type, tags, created_at
    """
    payload = {
        "query": query,
        "namespace": namespace,
        "top_k": top_k,
        "min_score": min_score
    }
    
    if memory_type:
        payload["type"] = memory_type
    if tags:
        payload["tags"] = tags
    if exclude_types:
        payload["exclude_types"] = exclude_types
    if date_range:
        payload["date_range"] = date_range
    
    response = requests.post(
        f"{CLOUD_RUN_URL}/searchMemory",
        json=payload,
        timeout=60
    )
    return response.json()

# Example: Search for protocols
results = search_memory(
    query="architecture decisions",
    namespace="claude",
    memory_type="emergent-moment",
    top_k=5
)

for match in results['data']['matches']:
    print(f"{match['id']}: {match['score']}")
    print(f"  {match['text'][:100]}...")
\`\`\`

### Fetch by ID

\`\`\`python
def fetch_memory(ids: list, namespace: str) -> dict:
    """Fetch specific memories by ID."""
    response = requests.post(
        f"{CLOUD_RUN_URL}/fetchMemory",
        json={"ids": ids, "namespace": namespace},
        timeout=30
    )
    return response.json()
\`\`\`

### Delete Memory

\`\`\`python
def delete_memory(ids: list = None, filter_dict: dict = None, namespace: str = "") -> dict:
    """Delete memories by ID or filter."""
    payload = {"namespace": namespace}
    if ids:
        payload["ids"] = ids
    if filter_dict:
        payload["filter"] = filter_dict
    
    response = requests.post(
        f"{CLOUD_RUN_URL}/deleteMemory",
        json=payload,
        timeout=30
    )
    return response.json()
\`\`\`

### Get Stats

\`\`\`python
def get_stats(namespace: str = None) -> dict:
    """Get index statistics."""
    response = requests.post(
        f"{CLOUD_RUN_URL}/describeStats",
        json={"namespace": namespace} if namespace else {},
        timeout=30
    )
    return response.json()
\`\`\`

---

## Quick Examples

### Store an emergent moment
\`\`\`python
store_memory(
    text="Realised that Option C (Overnight Claude) is optimal for budget analysis...",
    namespace="claude",
    memory_type="emergent-moment",
    tags=["architecture", "overnight-claude", "budget", "agentic-wisdom"]
)
\`\`\`

### Store a protocol
\`\`\`python
store_memory(
    text="Never apologise for being helpful. Confidence without arrogance.",
    namespace="claude",
    memory_type="protocol",
    tags=["behaviour", "confidence", "tone"]
)
\`\`\`

### Search your memories
\`\`\`python
results = search_memory(
    query="architecture decisions redis",
    namespace="claude",
    top_k=10
)
\`\`\`

### Search with filters
\`\`\`python
results = search_memory(
    query="important insights",
    namespace="claude",
    memory_type=["emergent-moment", "protocol"],
    date_range="30d",
    min_score=0.7
)
\`\`\`

---

## Current State

**Claude namespace created:** December 1, 2025

First entry: \`emergent-moment::emergent-moment-agentic-architecture-wisdom::20251201T043741Z\`
- Topic: Monzo budget architecture, Option C (Overnight Claude)
- Tagged: architecture, monzo, overnight-claude, redis, agentic-wisdom

---

## Notes

- The Cloud Run service handles OpenAI embeddings internally
- New types are accepted but flagged — stick to known types when possible
- Tags are freeform — use whatever helps semantic search
- Each namespace is isolated — Claude can't accidentally overwrite Moneypenny's memories
- The ID format \`{type}::{slug}::{timestamp}\` is auto-generated if not provided

---

*Skill rewritten December 1, 2025 to use Cloud Run API instead of direct Pinecone access.*
