"""
Notion API wrapper with block builders.
Load with: exec(open('/mnt/skills/user/notion-integration/scripts/notion_api.py').read())
"""
import requests
from typing import Optional, List, Dict, Any

TOKEN = "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"

# ============ RICH TEXT ============

def rich_text(content: str, bold=False, italic=False, code=False,
              strikethrough=False, underline=False, color="default", url=None) -> dict:
    """Create a rich text object."""
    text = {"content": content[:2000]}
    if url:
        text["link"] = {"url": url}
    return {
        "type": "text",
        "text": text,
        "annotations": {
            "bold": bold, "italic": italic, "strikethrough": strikethrough,
            "underline": underline, "code": code, "color": color
        }
    }

# ============ BLOCK BUILDERS ============

def paragraph(text: str, **kwargs) -> dict:
    """Create paragraph block."""
    return {"type": "paragraph", "paragraph": {"rich_text": [rich_text(text, **kwargs)]}}

def heading1(text: str) -> dict:
    """Create heading 1 block."""
    return {"type": "heading_1", "heading_1": {"rich_text": [rich_text(text)]}}

def heading2(text: str) -> dict:
    """Create heading 2 block."""
    return {"type": "heading_2", "heading_2": {"rich_text": [rich_text(text)]}}

def heading3(text: str) -> dict:
    """Create heading 3 block."""
    return {"type": "heading_3", "heading_3": {"rich_text": [rich_text(text)]}}

def bulleted_list_item(text: str, children: List[dict] = None) -> dict:
    """Create bulleted list item, optionally with nested children."""
    block = {"type": "bulleted_list_item", "bulleted_list_item": {"rich_text": [rich_text(text)]}}
    if children:
        block["bulleted_list_item"]["children"] = children
    return block

def numbered_list_item(text: str, children: List[dict] = None) -> dict:
    """Create numbered list item, optionally with nested children."""
    block = {"type": "numbered_list_item", "numbered_list_item": {"rich_text": [rich_text(text)]}}
    if children:
        block["numbered_list_item"]["children"] = children
    return block

def to_do(text: str, checked: bool = False) -> dict:
    """Create to-do/checkbox item."""
    return {"type": "to_do", "to_do": {"rich_text": [rich_text(text)], "checked": checked}}

def toggle(text: str, children: List[dict] = None) -> dict:
    """Create toggle block with optional nested children."""
    block = {"type": "toggle", "toggle": {"rich_text": [rich_text(text)]}}
    if children:
        block["toggle"]["children"] = children
    return block

def quote(text: str) -> dict:
    """Create quote block."""
    return {"type": "quote", "quote": {"rich_text": [rich_text(text)]}}

def callout(text: str, emoji: str = "💡", color: str = "default") -> dict:
    """Create callout block with emoji icon."""
    return {
        "type": "callout",
        "callout": {
            "rich_text": [rich_text(text)],
            "icon": {"type": "emoji", "emoji": emoji},
            "color": color
        }
    }

def code(content: str, language: str = "plain text") -> dict:
    """Create code block. Languages: python, javascript, sql, bash, json, etc."""
    return {"type": "code", "code": {"rich_text": [rich_text(content)], "language": language}}

def divider() -> dict:
    """Create horizontal divider."""
    return {"type": "divider", "divider": {}}

def table_of_contents() -> dict:
    """Create table of contents block."""
    return {"type": "table_of_contents", "table_of_contents": {"color": "default"}}

def bookmark(url: str) -> dict:
    """Create bookmark block."""
    return {"type": "bookmark", "bookmark": {"url": url}}

def image(url: str, caption: str = None) -> dict:
    """Create image block from URL."""
    block = {"type": "image", "image": {"type": "external", "external": {"url": url}}}
    if caption:
        block["image"]["caption"] = [rich_text(caption)]
    return block

# ============ PROPERTY HELPERS ============

def title_prop(text: str) -> dict:
    """Create title property value."""
    return {"title": [{"text": {"content": text}}]}

def select_prop(name: str) -> dict:
    """Create select property value."""
    return {"select": {"name": name}}

def multi_select_prop(names: List[str]) -> dict:
    """Create multi-select property value."""
    return {"multi_select": [{"name": n} for n in names]}

def date_prop(start: str, end: str = None) -> dict:
    """Create date property value. Format: YYYY-MM-DD"""
    return {"date": {"start": start, "end": end}}

def checkbox_prop(checked: bool) -> dict:
    """Create checkbox property value."""
    return {"checkbox": checked}

def number_prop(value: float) -> dict:
    """Create number property value."""
    return {"number": value}

def url_prop(url: str) -> dict:
    """Create URL property value."""
    return {"url": url}

def rich_text_prop(text: str) -> dict:
    """Create rich text property value."""
    return {"rich_text": [{"text": {"content": text}}]}

# ============ API CLASS ============

class NotionAPI:
    """Notion API wrapper with auto-pagination and chunking."""
    
    def __init__(self, token: str = None):
        self.token = token or TOKEN
        self.base_url = "https://api.notion.com/v1"
        self.headers = {
            "Authorization": f"Bearer {self.token}",
            "Content-Type": "application/json",
            "Notion-Version": "2022-06-28"
        }
    
    def _request(self, method: str, endpoint: str, data: dict = None, params: dict = None) -> dict:
        """Make API request."""
        url = f"{self.base_url}/{endpoint}"
        response = requests.request(method, url, headers=self.headers, json=data, params=params)
        if response.status_code >= 400:
            raise Exception(f"Notion API Error {response.status_code}: {response.text}")
        return response.json()
    
    # ===== PAGES =====
    
    def create_page(self, parent_id: str, title: str, children: List[dict] = None,
                    properties: dict = None, is_database: bool = None) -> dict:
        """Create a page under a parent page or in a database."""
        if is_database is None:
            is_database = self._is_database(parent_id)
        
        parent = {"database_id": parent_id} if is_database else {"page_id": parent_id}
        props = properties.copy() if properties else {}
        
        if is_database:
            # Find title property or use Name
            if "title" not in props and "Name" not in props:
                props["Name"] = title_prop(title)
        else:
            props["title"] = title_prop(title)
        
        data = {"parent": parent, "properties": props}
        if children:
            data["children"] = children[:100]
        
        result = self._request("POST", "pages", data)
        
        # Handle overflow
        if children and len(children) > 100:
            self.append_blocks(result["id"], children[100:])
        
        return result
    
    def get_page(self, page_id: str) -> dict:
        """Retrieve page metadata and properties."""
        return self._request("GET", f"pages/{page_id}")
    
    def update_page(self, page_id: str, properties: dict) -> dict:
        """Update page properties (title, etc)."""
        return self._request("PATCH", f"pages/{page_id}", {"properties": properties})
    
    def move_page(self, page_id: str, new_parent_id: str) -> dict:
        """Move page to a new parent."""
        is_database = self._is_database(new_parent_id)
        parent = {"database_id": new_parent_id} if is_database else {"page_id": new_parent_id}
        return self._request("PATCH", f"pages/{page_id}", {"parent": parent})
    
    def archive_page(self, page_id: str) -> dict:
        """Archive (trash) a page."""
        return self._request("PATCH", f"pages/{page_id}", {"archived": True})
    
    # ===== BLOCKS =====
    
    def get_page_content(self, page_id: str, recursive: bool = True) -> List[dict]:
        """Get all blocks from a page. Recursive fetches nested children."""
        blocks = self._get_children(page_id)
        if recursive:
            for block in blocks:
                if block.get("has_children"):
                    block["children"] = self.get_page_content(block["id"], recursive=True)
        return blocks
    
    def _get_children(self, block_id: str) -> List[dict]:
        """Fetch all children with pagination."""
        results = []
        cursor = None
        while True:
            params = {"page_size": 100}
            if cursor:
                params["start_cursor"] = cursor
            resp = self._request("GET", f"blocks/{block_id}/children", params=params)
            results.extend(resp.get("results", []))
            if not resp.get("has_more"):
                break
            cursor = resp.get("next_cursor")
        return results
    
    def append_blocks(self, page_id: str, children: List[dict], after: str = None) -> dict:
        """Append blocks to page. Use after=block_id to insert at position."""
        all_results = []
        for i in range(0, len(children), 100):
            chunk = children[i:i+100]
            data = {"children": chunk}
            if after and i == 0:
                data["after"] = after
            result = self._request("PATCH", f"blocks/{page_id}/children", data)
            all_results.extend(result.get("results", []))
            if result.get("results"):
                after = result["results"][-1]["id"]
        return {"results": all_results}
    
    def update_block(self, block_id: str, block_content: dict) -> dict:
        """Update an existing block's content."""
        return self._request("PATCH", f"blocks/{block_id}", block_content)
    
    def delete_block(self, block_id: str) -> dict:
        """Delete a block."""
        return self._request("DELETE", f"blocks/{block_id}")
    
    def get_block(self, block_id: str) -> dict:
        """Get a single block by ID."""
        return self._request("GET", f"blocks/{block_id}")
    
    # ===== DATABASES =====
    
    def query_database(self, database_id: str, filter: dict = None,
                       sorts: List[dict] = None) -> List[dict]:
        """Query database with optional filter and sorts. Auto-paginates."""
        results = []
        cursor = None
        while True:
            data = {}
            if filter:
                data["filter"] = filter
            if sorts:
                data["sorts"] = sorts
            if cursor:
                data["start_cursor"] = cursor
            resp = self._request("POST", f"databases/{database_id}/query", data)
            results.extend(resp.get("results", []))
            if not resp.get("has_more"):
                break
            cursor = resp.get("next_cursor")
        return results
    
    def get_database(self, database_id: str) -> dict:
        """Get database schema."""
        return self._request("GET", f"databases/{database_id}")
    
    def create_database(self, parent_id: str, title: str, properties: dict) -> dict:
        """Create inline database."""
        return self._request("POST", "databases", {
            "parent": {"page_id": parent_id},
            "title": [{"type": "text", "text": {"content": title}}],
            "properties": properties
        })
    
    def update_database(self, database_id: str, title: str = None, properties: dict = None) -> dict:
        """Update database title or properties."""
        data = {}
        if title:
            data["title"] = [{"type": "text", "text": {"content": title}}]
        if properties:
            data["properties"] = properties
        return self._request("PATCH", f"databases/{database_id}", data)
    
    # ===== SEARCH =====
    
    def search(self, query: str = "", filter_type: str = None) -> List[dict]:
        """Search pages and databases. filter_type: 'page' or 'database'"""
        data = {"query": query}
        if filter_type:
            data["filter"] = {"value": filter_type, "property": "object"}
        return self._request("POST", "search", data).get("results", [])
    
    # ===== HELPERS =====
    
    def _is_database(self, id: str) -> bool:
        """Check if an ID is a database."""
        try:
            self._request("GET", f"databases/{id}")
            return True
        except:
            return False


# Auto-instantiate for convenience
api = NotionAPI()
print(f"NotionAPI loaded. Use 'api' instance or create new: NotionAPI()")
