# Database Query Filters

Complete filter syntax for database queries.

## Basic Structure

```python
# Single condition
filter = {
    "property": "Status",
    "select": {"equals": "Done"}
}

# Multiple conditions (AND)
filter = {
    "and": [
        {"property": "Status", "select": {"equals": "Done"}},
        {"property": "Priority", "number": {"greater_than": 2}}
    ]
}

# Multiple conditions (OR)
filter = {
    "or": [
        {"property": "Status", "select": {"equals": "Done"}},
        {"property": "Status", "select": {"equals": "In Progress"}}
    ]
}

# Nested logic
filter = {
    "and": [
        {"property": "Active", "checkbox": {"equals": True}},
        {
            "or": [
                {"property": "Priority", "select": {"equals": "High"}},
                {"property": "Urgent", "checkbox": {"equals": True}}
            ]
        }
    ]
}
```

---

## Filter by Property Type

### Title / Rich Text / URL / Email / Phone
```python
{"property": "Name", "title": {"equals": "Exact Match"}}
{"property": "Name", "title": {"contains": "substring"}}
{"property": "Name", "title": {"does_not_contain": "excluded"}}
{"property": "Name", "title": {"starts_with": "prefix"}}
{"property": "Name", "title": {"ends_with": "suffix"}}
{"property": "Name", "title": {"is_empty": True}}
{"property": "Name", "title": {"is_not_empty": True}}

# Same for rich_text, url, email, phone_number
{"property": "Notes", "rich_text": {"contains": "keyword"}}
{"property": "Website", "url": {"contains": "example.com"}}
```

### Number
```python
{"property": "Count", "number": {"equals": 5}}
{"property": "Count", "number": {"does_not_equal": 0}}
{"property": "Count", "number": {"greater_than": 10}}
{"property": "Count", "number": {"greater_than_or_equal_to": 10}}
{"property": "Count", "number": {"less_than": 100}}
{"property": "Count", "number": {"less_than_or_equal_to": 100}}
{"property": "Count", "number": {"is_empty": True}}
{"property": "Count", "number": {"is_not_empty": True}}
```

### Select
```python
{"property": "Status", "select": {"equals": "Done"}}
{"property": "Status", "select": {"does_not_equal": "Cancelled"}}
{"property": "Status", "select": {"is_empty": True}}
{"property": "Status", "select": {"is_not_empty": True}}
```

### Multi-Select
```python
{"property": "Tags", "multi_select": {"contains": "Important"}}
{"property": "Tags", "multi_select": {"does_not_contain": "Archive"}}
{"property": "Tags", "multi_select": {"is_empty": True}}
{"property": "Tags", "multi_select": {"is_not_empty": True}}
```

### Checkbox
```python
{"property": "Done", "checkbox": {"equals": True}}
{"property": "Done", "checkbox": {"equals": False}}
```

### Date
```python
# Exact date
{"property": "Due", "date": {"equals": "2025-01-15"}}
{"property": "Due", "date": {"before": "2025-01-01"}}
{"property": "Due", "date": {"after": "2025-01-01"}}
{"property": "Due", "date": {"on_or_before": "2025-12-31"}}
{"property": "Due", "date": {"on_or_after": "2025-01-01"}}

# Relative dates
{"property": "Due", "date": {"past_week": {}}}
{"property": "Due", "date": {"past_month": {}}}
{"property": "Due", "date": {"past_year": {}}}
{"property": "Due", "date": {"next_week": {}}}
{"property": "Due", "date": {"next_month": {}}}
{"property": "Due", "date": {"next_year": {}}}
{"property": "Due", "date": {"this_week": {}}}

# Empty checks
{"property": "Due", "date": {"is_empty": True}}
{"property": "Due", "date": {"is_not_empty": True}}
```

### People
```python
{"property": "Assignee", "people": {"contains": "user-uuid"}}
{"property": "Assignee", "people": {"does_not_contain": "user-uuid"}}
{"property": "Assignee", "people": {"is_empty": True}}
{"property": "Assignee", "people": {"is_not_empty": True}}
```

### Relation
```python
{"property": "Project", "relation": {"contains": "page-uuid"}}
{"property": "Project", "relation": {"does_not_contain": "page-uuid"}}
{"property": "Project", "relation": {"is_empty": True}}
{"property": "Project", "relation": {"is_not_empty": True}}
```

### Formula
Filter based on formula result type:
```python
# String formula
{"property": "Formula", "formula": {"string": {"contains": "text"}}}

# Number formula
{"property": "Formula", "formula": {"number": {"greater_than": 10}}}

# Checkbox formula
{"property": "Formula", "formula": {"checkbox": {"equals": True}}}

# Date formula
{"property": "Formula", "formula": {"date": {"after": "2025-01-01"}}}
```

### Rollup
Filter based on rollup type:
```python
# Number rollup
{"property": "Total", "rollup": {"number": {"greater_than": 100}}}

# Date rollup
{"property": "Latest", "rollup": {"date": {"after": "2025-01-01"}}}

# Array rollup (any/every/none)
{"property": "Items", "rollup": {"any": {"rich_text": {"contains": "keyword"}}}}
{"property": "Items", "rollup": {"every": {"checkbox": {"equals": True}}}}
{"property": "Items", "rollup": {"none": {"select": {"equals": "Cancelled"}}}}
```

### Created/Edited Time
```python
{"timestamp": "created_time", "created_time": {"after": "2025-01-01"}}
{"timestamp": "last_edited_time", "last_edited_time": {"past_week": {}}}
```

### Created/Edited By
```python
{"property": "Created By", "created_by": {"contains": "user-uuid"}}
{"property": "Last Edited By", "last_edited_by": {"contains": "user-uuid"}}
```

---

## Sorts

```python
# Single sort
sorts = [{"property": "Due Date", "direction": "ascending"}]

# Multiple sorts
sorts = [
    {"property": "Priority", "direction": "descending"},
    {"property": "Name", "direction": "ascending"}
]

# Sort by timestamp
sorts = [{"timestamp": "created_time", "direction": "descending"}]
sorts = [{"timestamp": "last_edited_time", "direction": "ascending"}]
```

Directions: `ascending`, `descending`

---

## Complete Example

```python
results = api.query_database(
    "database-id",
    filter={
        "and": [
            {"property": "Status", "select": {"does_not_equal": "Done"}},
            {"property": "Due", "date": {"on_or_before": "2025-01-31"}},
            {"property": "Priority", "number": {"greater_than_or_equal_to": 3}}
        ]
    },
    sorts=[
        {"property": "Priority", "direction": "descending"},
        {"property": "Due", "direction": "ascending"}
    ]
)
```
