# Advanced Notion Operations

Edge cases and complex patterns.

## Insert Blocks at Specific Position

The `after` parameter inserts new blocks AFTER the specified block_id.

```python
# 1. Get current blocks to find insertion point
blocks = api.get_page_content("page-id")

# Find block to insert after (e.g., after the 3rd block)
target_block_id = blocks[2]["id"]

# 2. Insert after that block
api.append_blocks("page-id", [
    paragraph("This appears after the 3rd block")
], after=target_block_id)
```

**Limitation:** No "before" parameter exists. To insert at the very beginning:
1. Get all blocks
2. Delete them all
3. Prepend your new content
4. Re-append the original blocks

```python
# Insert at beginning (destructive)
blocks = api.get_page_content("page-id")
for block in blocks:
    api.delete_block(block["id"])

new_content = [paragraph("This is now first")]
new_content.extend(blocks)  # Loses block IDs but keeps content
api.append_blocks("page-id", new_content)
```

---

## Deep Nesting (>2 levels)

API limits to 2 nesting levels per request. For deeper:

```python
# Step 1: Create parent toggle
result = api.append_blocks("page-id", [
    toggle("Level 1")
])
level1_id = result["results"][0]["id"]

# Step 2: Add level 2 inside level 1
result = api.append_blocks(level1_id, [
    toggle("Level 2")
])
level2_id = result["results"][0]["id"]

# Step 3: Add level 3 inside level 2
api.append_blocks(level2_id, [
    paragraph("Level 3 content")
])
```

---

## Synced Blocks

### Create Original (Source)
```python
synced = {
    "type": "synced_block",
    "synced_block": {
        "synced_from": None,
        "children": [
            paragraph("This content syncs everywhere"),
            callout("Shared callout", emoji="🔄")
        ]
    }
}
result = api.append_blocks("page-id", [synced])
original_id = result["results"][0]["id"]
```

### Create Reference (Duplicate)
```python
reference = {
    "type": "synced_block",
    "synced_block": {
        "synced_from": {"block_id": original_id}
    }
}
api.append_blocks("other-page-id", [reference])
```

Changes to original automatically sync to all references.

---

## Tables

### Create Table
```python
table = {
    "type": "table",
    "table": {
        "table_width": 3,
        "has_column_header": True,
        "has_row_header": False,
        "children": [
            {
                "type": "table_row",
                "table_row": {
                    "cells": [
                        [{"type": "text", "text": {"content": "Name"}}],
                        [{"type": "text", "text": {"content": "Status"}}],
                        [{"type": "text", "text": {"content": "Date"}}]
                    ]
                }
            },
            {
                "type": "table_row",
                "table_row": {
                    "cells": [
                        [{"type": "text", "text": {"content": "Task 1"}}],
                        [{"type": "text", "text": {"content": "Done"}}],
                        [{"type": "text", "text": {"content": "2025-01-15"}}]
                    ]
                }
            }
        ]
    }
}
api.append_blocks("page-id", [table])
```

---

## Column Layouts

```python
columns = {
    "type": "column_list",
    "column_list": {
        "children": [
            {
                "type": "column",
                "column": {
                    "children": [
                        heading2("Left Column"),
                        paragraph("Left content")
                    ]
                }
            },
            {
                "type": "column",
                "column": {
                    "children": [
                        heading2("Right Column"),
                        paragraph("Right content")
                    ]
                }
            }
        ]
    }
}
api.append_blocks("page-id", [columns])
```

---

## Handling Large Content

### Text >2000 Characters
Split into multiple rich_text objects:
```python
long_text = "..." * 3000  # 3000 chars

# Split into chunks
chunks = [long_text[i:i+2000] for i in range(0, len(long_text), 2000)]

block = {
    "type": "paragraph",
    "paragraph": {
        "rich_text": [{"type": "text", "text": {"content": chunk}} for chunk in chunks]
    }
}
```

### >100 Blocks
The API class handles this automatically, but manually:
```python
all_blocks = [...]  # 250 blocks

for i in range(0, len(all_blocks), 100):
    chunk = all_blocks[i:i+100]
    api.append_blocks("page-id", chunk)
```

---

## Reading Nested Content

`get_page_content` is recursive by default:
```python
blocks = api.get_page_content("page-id")

for block in blocks:
    print(block["type"])
    if "children" in block:
        for child in block["children"]:
            print(f"  - {child['type']}")
```

To skip nested fetching (faster):
```python
blocks = api.get_page_content("page-id", recursive=False)
```

---

## Update Block Content

To update existing text:
```python
# Get block's current type
block = api.get_block("block-id")
block_type = block["type"]  # e.g., "paragraph"

# Update with same structure
api.update_block("block-id", {
    block_type: {
        "rich_text": [{"type": "text", "text": {"content": "New content"}}]
    }
})
```

---

## Database Property Schemas

When creating a database:
```python
api.create_database("parent-page-id", "My Database", {
    "Name": {"title": {}},
    "Status": {
        "select": {
            "options": [
                {"name": "To Do", "color": "gray"},
                {"name": "In Progress", "color": "yellow"},
                {"name": "Done", "color": "green"}
            ]
        }
    },
    "Priority": {"number": {"format": "number"}},
    "Due Date": {"date": {}},
    "Tags": {
        "multi_select": {
            "options": [
                {"name": "Urgent", "color": "red"},
                {"name": "Important", "color": "orange"}
            ]
        }
    },
    "Completed": {"checkbox": {}},
    "Notes": {"rich_text": {}},
    "URL": {"url": {}},
    "Assignee": {"people": {}},
    "Related": {"relation": {"database_id": "other-db-id", "single_property": {}}}
})
```
