---
name: notion-integration
description: Creates, reads, and updates Notion pages, blocks, and databases. Use for any Notion workspace operations including creating pages, appending content, searching, querying databases, and managing block content.
---

# Notion Integration

## Configuration

**Token:** `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`

**Key Page IDs:**
- Chez Claude: `25cb3682-1109-809d-b169-ed9b20479ed8`
- The Bridge: `210b3682-1109-8085-945c-fda346ccb6c8`

## Quick Start

```python
exec(open('/mnt/skills/user/notion-integration/scripts/notion_api.py').read())

api = NotionAPI()

# Create a page under Chez Claude
page = api.create_page(
    parent_id="25cb3682-1109-809d-b169-ed9b20479ed8",
    title="My New Page",
    children=[
        heading2("Overview"),
        paragraph("This is the content."),
        callout("Important note", emoji="⚠️")
    ]
)
print(f"Created: {page['url']}")
```

---

## Block Builders

All builders are available after loading the script.

### Text Blocks
```python
paragraph("Your text here")
paragraph("Bold text", bold=True)
heading1("Main Heading")
heading2("Section Heading")
heading3("Subsection")
quote("A memorable quote")
```

### Callouts
```python
callout("Note text", emoji="💡")
callout("Warning", emoji="⚠️", color="yellow_background")
callout("Success", emoji="✅", color="green_background")
```

### Lists
```python
bulleted_list_item("First point")
bulleted_list_item("With children", children=[
    paragraph("Nested content")
])
numbered_list_item("Step one")
to_do("Task item", checked=False)
```

### Special Blocks
```python
toggle("Click to expand", children=[
    paragraph("Hidden content")
])
code("print('hello')", language="python")
divider()
```

### Nesting Example
```python
children = [
    heading2("Section"),
    paragraph("Intro text"),
    callout("Key point", emoji="🎯", color="blue_background"),
    toggle("Details", children=[
        paragraph("Expanded info"),
        bulleted_list_item("Point 1"),
        bulleted_list_item("Point 2")
    ]),
    divider()
]
```

---

## Common Operations

### Create Page
```python
# Under a page (like Chez Claude)
api.create_page(
    parent_id="25cb3682-1109-809d-b169-ed9b20479ed8",
    title="Page Title",
    children=[paragraph("Content")]
)

# In a database (properties vary by schema)
api.create_page(
    parent_id="database-id",
    title="Entry Name",
    properties={
        "Status": select_prop("Active"),
        "Priority": {"number": 1}
    }
)
```

### Read Page Content
```python
# Get all blocks (auto-paginates, fetches nested)
blocks = api.get_page_content("page-id")

for block in blocks:
    print(block['type'], block.get('has_children'))
```

### Append to Existing Page
```python
# Add to end
api.append_blocks("page-id", [
    heading2("New Section"),
    paragraph("Additional content")
])

# Insert after specific block
api.append_blocks("page-id", [
    paragraph("Inserted content")
], after="block-id-to-insert-after")
```

### Update Page Title
```python
api.update_page("page-id", properties={
    "title": title_prop("New Title")
})
```

### Move Page to Different Parent
```python
api.move_page("page-id", new_parent_id="new-parent-page-id")
```

### Delete/Archive
```python
api.delete_block("block-id")  # Remove a block
api.archive_page("page-id")   # Move page to trash
```

### Search
```python
# Find pages
results = api.search("project name")

# Filter to just pages or databases
pages = api.search("query", filter_type="page")
databases = api.search("query", filter_type="database")
```

### Query Database
```python
# Simple query
items = api.query_database("database-id")

# With filter
items = api.query_database("database-id", filter={
    "property": "Status",
    "select": {"equals": "Active"}
})

# Multiple conditions
items = api.query_database("database-id", filter={
    "and": [
        {"property": "Status", "select": {"equals": "Done"}},
        {"property": "Priority", "number": {"greater_than": 2}}
    ]
})

# With sort
items = api.query_database("database-id", 
    sorts=[{"property": "Date", "direction": "descending"}]
)
```

---

## Property Helpers

For creating pages in databases:

```python
properties = {
    "Name": title_prop("Page Title"),           # Title (required)
    "Status": select_prop("In Progress"),        # Single select
    "Tags": multi_select_prop(["AI", "Project"]),# Multi select
    "Due": date_prop("2025-01-15"),              # Date
    "Priority": {"number": 1},                   # Number
    "Done": checkbox_prop(True),                 # Checkbox
    "URL": {"url": "https://example.com"}        # URL
}
```

---

## API Methods Reference

### Pages
| Method | Description |
|--------|-------------|
| `create_page(parent_id, title, children=[], properties={})` | Create page |
| `get_page(page_id)` | Get page metadata |
| `update_page(page_id, properties)` | Update properties |
| `move_page(page_id, new_parent_id)` | Move to new parent |
| `archive_page(page_id)` | Move to trash |

### Blocks
| Method | Description |
|--------|-------------|
| `get_page_content(page_id)` | Get all blocks (recursive) |
| `append_blocks(page_id, children, after=None)` | Add blocks |
| `update_block(block_id, content)` | Modify block |
| `delete_block(block_id)` | Remove block |

### Databases
| Method | Description |
|--------|-------------|
| `query_database(db_id, filter=None, sorts=None)` | Query with filters |
| `get_database(db_id)` | Get schema |
| `create_database(parent_id, title, properties)` | Create database |

### Search
| Method | Description |
|--------|-------------|
| `search(query, filter_type=None)` | Find pages/databases |

---

## Constraints

- **100 blocks max** per request (auto-chunked)
- **2 nesting levels** per request
- **2,000 chars max** per text block
- **Rate limit:** 3 req/sec average

---

## Advanced Operations

For complex operations, see:
- `references/block-types.md` - All 27 block type templates
- `references/database-filters.md` - Complete filter syntax
- `references/advanced-operations.md` - Synced blocks, tables, columns
