"""
Margo PA API - Claude's Personal Assistant.
Load with: exec(open('/mnt/skills/user/margo/scripts/margo_api.py').read())

Direct HTTPS to margo.willcureton.com - no intermediaries.
"""
import requests
from typing import Dict, Any

BASE_URL = "https://margo.willcureton.com"


class Margo:
    """Claude's Personal Assistant - contextual briefings and questions."""
    
    def __init__(self, base_url: str = None):
        self.base_url = base_url or BASE_URL
    
    def _get(self, endpoint: str) -> Dict[str, Any]:
        """GET request to Margo."""
        response = requests.get(f"{self.base_url}{endpoint}", timeout=90)
        response.raise_for_status()
        return response.json()
    
    def _post(self, endpoint: str, data: Dict = None) -> Dict[str, Any]:
        """POST request to Margo."""
        response = requests.post(
            f"{self.base_url}{endpoint}",
            json=data or {},
            headers={"Content-Type": "application/json"},
            timeout=90
        )
        response.raise_for_status()
        return response.json()
    
    # ===== PUBLIC API =====
    
    def brief(self) -> Dict[str, Any]:
        """
        Get full contextual briefing.
        
        Returns:
            dict with briefing text, project, skills_count, scenario, context
        """
        return self._get("/brief")
    
    def ask(self, question: str) -> str:
        """
        Ask Margo a question.
        
        Args:
            question: What you want to know
        
        Returns:
            str: Margo's answer
        """
        result = self._post("/ask", {"question": question})
        return result.get("answer", "")
    
    def health(self) -> Dict[str, Any]:
        """
        Check service health.
        
        Returns:
            dict with status, model_loaded, cache_entries, integrations
        """
        return self._get("/health")
    
    def context(self) -> Dict[str, Any]:
        """
        Get raw context data (debug).
        
        Returns:
            dict with all context Margo has about Will
        """
        return self._get("/context")
    
    def clear_cache(self) -> Dict[str, Any]:
        """
        Force fresh briefing on next call.
        
        Returns:
            dict with status confirmation
        """
        return self._post("/clear-cache")


# Auto-instantiate
margo = Margo()
print("Margo PA ready (direct HTTPS)")
print("Use 'margo' instance: brief(), ask(), health()")
