"""
Cloudflare API wrapper - FULL ACCESS (God Mode 😈)
Load with: exec(open('/mnt/skills/user/cloudflare-dns/scripts/cloudflare_api.py').read())

Capabilities:
- DNS management
- Firewall rules & WAF
- Page rules & redirects
- SSL/TLS settings
- Cache control
- Zone settings (56 options)
- API token management
"""
import requests
from typing import Optional, List, Dict, Any

TOKEN = "Kc845neMXGbNBvPM-ol9ZuXFwGgCWjOlCm6vEtxS"

# Known zones for quick access
ZONES = {
    "willcureton.com": "86b4c6bfc8d2e6ce3a6e715f89ce27a6",
    "willcureton.co.uk": "ffeacda58b722365873791775dd99779"
}

# VPS IP for quick reference
VPS_IP = "65.21.178.196"


class CloudflareAPI:
    """Cloudflare API wrapper for DNS management."""
    
    def __init__(self, token: str = None):
        self.token = token or TOKEN
        self.base_url = "https://api.cloudflare.com/client/v4"
        self.headers = {
            "Authorization": f"Bearer {self.token}",
            "Content-Type": "application/json"
        }
        self._zone_cache = ZONES.copy()
    
    def _request(self, method: str, endpoint: str, data: dict = None, params: dict = None) -> dict:
        """Make API request."""
        url = f"{self.base_url}/{endpoint}"
        response = requests.request(method, url, headers=self.headers, json=data, params=params)
        result = response.json()
        if not result.get("success", False):
            errors = result.get("errors", [])
            raise Exception(f"Cloudflare API Error: {errors}")
        return result
    
    # ===== ZONES =====
    
    def list_zones(self) -> List[dict]:
        """List all zones accessible with this token."""
        result = self._request("GET", "zones")
        zones = result.get("result", [])
        # Update cache
        for z in zones:
            self._zone_cache[z["name"]] = z["id"]
        return zones
    
    def get_zone(self, zone_id: str) -> dict:
        """Get zone details by ID."""
        return self._request("GET", f"zones/{zone_id}").get("result")
    
    def get_zone_id(self, domain: str) -> str:
        """Get zone ID for a domain name. Uses cache first."""
        if domain in self._zone_cache:
            return self._zone_cache[domain]
        # Fetch from API
        result = self._request("GET", "zones", params={"name": domain})
        zones = result.get("result", [])
        if zones:
            zone_id = zones[0]["id"]
            self._zone_cache[domain] = zone_id
            return zone_id
        raise Exception(f"Zone not found: {domain}")
    
    def purge_cache(self, domain: str, purge_everything: bool = True, 
                    files: List[str] = None, tags: List[str] = None) -> dict:
        """Purge zone cache. By default purges everything."""
        zone_id = self.get_zone_id(domain)
        data = {}
        if purge_everything:
            data["purge_everything"] = True
        elif files:
            data["files"] = files
        elif tags:
            data["tags"] = tags
        return self._request("POST", f"zones/{zone_id}/purge_cache", data).get("result")
    
    # ===== DNS RECORDS =====
    
    def list_records(self, domain: str, record_type: str = None, 
                     name: str = None) -> List[dict]:
        """List DNS records for a domain. Optionally filter by type or name."""
        zone_id = self.get_zone_id(domain)
        params = {}
        if record_type:
            params["type"] = record_type
        if name:
            params["name"] = name if "." in name else f"{name}.{domain}"
        result = self._request("GET", f"zones/{zone_id}/dns_records", params=params)
        return result.get("result", [])
    
    def get_record(self, domain: str, record_id: str) -> dict:
        """Get a specific DNS record by ID."""
        zone_id = self.get_zone_id(domain)
        return self._request("GET", f"zones/{zone_id}/dns_records/{record_id}").get("result")
    
    def create_record(self, domain: str, name: str, record_type: str, content: str,
                      proxied: bool = False, ttl: int = 1, priority: int = None,
                      comment: str = None) -> dict:
        """
        Create a DNS record.
        
        Args:
            domain: The zone (e.g., 'willcureton.com')
            name: Subdomain or '@' for root (e.g., 'mcp' for mcp.willcureton.com)
            record_type: A, AAAA, CNAME, TXT, MX, etc.
            content: IP address or value
            proxied: True for orange cloud (CDN), False for grey (DNS only)
            ttl: 1 for auto, or seconds (60-86400)
            priority: Required for MX records (lower = higher priority)
            comment: Optional comment for the record
        
        Returns:
            Created record dict
        """
        zone_id = self.get_zone_id(domain)
        
        # Handle @ for root
        full_name = domain if name == "@" else f"{name}.{domain}" if "." not in name else name
        
        data = {
            "type": record_type.upper(),
            "name": full_name,
            "content": content,
            "ttl": ttl,
            "proxied": proxied if record_type.upper() in ["A", "AAAA", "CNAME"] else False
        }
        
        if priority is not None:
            data["priority"] = priority
        if comment:
            data["comment"] = comment
        
        return self._request("POST", f"zones/{zone_id}/dns_records", data).get("result")
    
    def update_record(self, domain: str, record_id: str, name: str = None,
                      record_type: str = None, content: str = None,
                      proxied: bool = None, ttl: int = None,
                      comment: str = None) -> dict:
        """
        Update an existing DNS record. Only provided fields are updated.
        """
        zone_id = self.get_zone_id(domain)
        
        # Get current record
        current = self.get_record(domain, record_id)
        
        data = {
            "type": record_type.upper() if record_type else current["type"],
            "name": name if name else current["name"],
            "content": content if content else current["content"],
            "ttl": ttl if ttl is not None else current["ttl"],
            "proxied": proxied if proxied is not None else current.get("proxied", False)
        }
        
        if comment:
            data["comment"] = comment
        
        return self._request("PUT", f"zones/{zone_id}/dns_records/{record_id}", data).get("result")
    
    def delete_record(self, domain: str, record_id: str) -> dict:
        """Delete a DNS record."""
        zone_id = self.get_zone_id(domain)
        return self._request("DELETE", f"zones/{zone_id}/dns_records/{record_id}").get("result")
    
    def find_record(self, domain: str, name: str, record_type: str = None) -> Optional[dict]:
        """Find a record by name (and optionally type). Returns first match or None."""
        records = self.list_records(domain, record_type=record_type, name=name)
        return records[0] if records else None
    
    # ===== CONVENIENCE METHODS =====
    
    def add_subdomain(self, domain: str, subdomain: str, ip: str = None,
                      proxied: bool = False, comment: str = None) -> dict:
        """
        Quick method to add a subdomain pointing to the VPS.
        
        Args:
            domain: Base domain (default: willcureton.com)
            subdomain: Subdomain name (e.g., 'api', 'mcp')
            ip: IP address (default: VPS_IP)
            proxied: Enable Cloudflare proxy (orange cloud)
            comment: Optional description
        
        Returns:
            Created record
        """
        return self.create_record(
            domain=domain,
            name=subdomain,
            record_type="A",
            content=ip or VPS_IP,
            proxied=proxied,
            comment=comment
        )
    
    def add_cname(self, domain: str, name: str, target: str,
                  proxied: bool = False, comment: str = None) -> dict:
        """Add a CNAME record."""
        return self.create_record(
            domain=domain,
            name=name,
            record_type="CNAME",
            content=target,
            proxied=proxied,
            comment=comment
        )
    
    def add_txt(self, domain: str, name: str, content: str, comment: str = None) -> dict:
        """Add a TXT record (for verification, SPF, etc.)."""
        return self.create_record(
            domain=domain,
            name=name,
            record_type="TXT",
            content=content,
            comment=comment
        )
    
    def add_mx(self, domain: str, name: str, mailserver: str, 
               priority: int = 10, comment: str = None) -> dict:
        """Add an MX record for email routing."""
        return self.create_record(
            domain=domain,
            name=name,
            record_type="MX",
            content=mailserver,
            priority=priority,
            comment=comment
        )
    
    def toggle_proxy(self, domain: str, name: str, proxied: bool) -> dict:
        """Toggle Cloudflare proxy (orange/grey cloud) for a record."""
        record = self.find_record(domain, name)
        if not record:
            raise Exception(f"Record not found: {name}.{domain}")
        return self.update_record(domain, record["id"], proxied=proxied)
    
    def update_ip(self, domain: str, name: str, new_ip: str) -> dict:
        """Update the IP address for an A record."""
        record = self.find_record(domain, name, record_type="A")
        if not record:
            raise Exception(f"A record not found: {name}.{domain}")
        return self.update_record(domain, record["id"], content=new_ip)
    
    # ===== BULK OPERATIONS =====
    
    def export_records(self, domain: str) -> str:
        """Export all DNS records as BIND format."""
        zone_id = self.get_zone_id(domain)
        url = f"{self.base_url}/zones/{zone_id}/dns_records/export"
        response = requests.get(url, headers=self.headers)
        return response.text
    
    def list_all_subdomains(self, domain: str) -> List[str]:
        """Get list of all subdomains for a domain."""
        records = self.list_records(domain)
        subdomains = set()
        for r in records:
            name = r["name"]
            if name != domain:
                subdomain = name.replace(f".{domain}", "")
                subdomains.add(subdomain)
        return sorted(list(subdomains))
    
    # ===== ZONE SETTINGS =====
    
    def get_settings(self, domain: str) -> List[dict]:
        """Get all zone settings."""
        zone_id = self.get_zone_id(domain)
        return self._request("GET", f"zones/{zone_id}/settings").get("result", [])
    
    def update_setting(self, domain: str, setting: str, value: Any) -> dict:
        """Update a zone setting (e.g., ssl, minify, always_https)."""
        zone_id = self.get_zone_id(domain)
        return self._request("PATCH", f"zones/{zone_id}/settings/{setting}", 
                            {"value": value}).get("result")
    
    def enable_https(self, domain: str) -> dict:
        """Enable Always Use HTTPS."""
        return self.update_setting(domain, "always_use_https", "on")
    
    def set_ssl_mode(self, domain: str, mode: str = "full") -> dict:
        """Set SSL mode: off, flexible, full, strict."""
        return self.update_setting(domain, "ssl", mode)
    
    # ===== ANALYTICS =====
    
    def get_analytics(self, domain: str, since: str = "-1440") -> dict:
        """
        Get zone analytics.
        since: Minutes ago (e.g., "-1440" for last 24h, "-10080" for last week)
        """
        zone_id = self.get_zone_id(domain)
        params = {"since": since}
        return self._request("GET", f"zones/{zone_id}/analytics/dashboard", 
                            params=params).get("result")
    
    # ===== FIREWALL RULES =====
    
    def list_firewall_rules(self, domain: str) -> List[dict]:
        """List all firewall rules for a zone."""
        zone_id = self.get_zone_id(domain)
        return self._request("GET", f"zones/{zone_id}/firewall/rules").get("result", [])
    
    def create_firewall_rule(self, domain: str, expression: str, action: str,
                             description: str = None, priority: int = None,
                             paused: bool = False) -> dict:
        """
        Create a firewall rule.
        
        Args:
            domain: Zone domain
            expression: Filter expression (e.g., 'ip.src == 1.2.3.4')
            action: block, challenge, js_challenge, managed_challenge, allow, log, bypass
            description: Rule description
            priority: Lower = higher priority
            paused: Whether rule is paused
        
        Expression examples:
            'ip.src == 1.2.3.4'
            'ip.src in {1.2.3.4 5.6.7.8}'
            'http.request.uri.path contains "/admin"'
            'cf.threat_score > 10'
            '(ip.geoip.country == "RU") or (ip.geoip.country == "CN")'
        """
        zone_id = self.get_zone_id(domain)
        
        # Create filter first
        filter_data = {"expression": expression, "paused": paused}
        if description:
            filter_data["description"] = description
        
        filter_result = self._request("POST", f"zones/{zone_id}/filters", [filter_data])
        filter_id = filter_result["result"][0]["id"]
        
        # Create rule using filter
        rule_data = {
            "filter": {"id": filter_id},
            "action": action,
            "paused": paused
        }
        if description:
            rule_data["description"] = description
        if priority:
            rule_data["priority"] = priority
            
        return self._request("POST", f"zones/{zone_id}/firewall/rules", [rule_data]).get("result", [{}])[0]
    
    def delete_firewall_rule(self, domain: str, rule_id: str) -> dict:
        """Delete a firewall rule."""
        zone_id = self.get_zone_id(domain)
        return self._request("DELETE", f"zones/{zone_id}/firewall/rules/{rule_id}").get("result")
    
    def block_ip(self, domain: str, ip: str, note: str = None) -> dict:
        """Quick method to block an IP address."""
        desc = note or f"Blocked {ip}"
        return self.create_firewall_rule(domain, f'ip.src == {ip}', "block", description=desc)
    
    def block_country(self, domain: str, country_code: str, note: str = None) -> dict:
        """Block traffic from a country (2-letter code, e.g., 'RU', 'CN')."""
        desc = note or f"Blocked country: {country_code}"
        return self.create_firewall_rule(domain, f'ip.geoip.country == "{country_code}"', "block", description=desc)
    
    def challenge_high_threat(self, domain: str, threshold: int = 10) -> dict:
        """Challenge requests with high threat score."""
        return self.create_firewall_rule(
            domain, 
            f'cf.threat_score > {threshold}', 
            "managed_challenge",
            description=f"Challenge threat score > {threshold}"
        )
    
    # ===== ACCESS RULES (IP blocking at account level) =====
    
    def list_access_rules(self, domain: str) -> List[dict]:
        """List zone-level access rules (IP/country blocks)."""
        zone_id = self.get_zone_id(domain)
        return self._request("GET", f"zones/{zone_id}/firewall/access_rules/rules").get("result", [])
    
    def create_access_rule(self, domain: str, mode: str, value: str, 
                          target: str = "ip", note: str = None) -> dict:
        """
        Create an access rule.
        
        Args:
            mode: block, challenge, whitelist, js_challenge, managed_challenge
            value: IP, CIDR, country code, or ASN
            target: ip, ip_range, country, asn
            note: Description
        """
        zone_id = self.get_zone_id(domain)
        data = {
            "mode": mode,
            "configuration": {"target": target, "value": value}
        }
        if note:
            data["notes"] = note
        return self._request("POST", f"zones/{zone_id}/firewall/access_rules/rules", data).get("result")
    
    def whitelist_ip(self, domain: str, ip: str, note: str = None) -> dict:
        """Whitelist an IP address."""
        return self.create_access_rule(domain, "whitelist", ip, "ip", note or f"Whitelisted {ip}")
    
    # ===== PAGE RULES =====
    
    def list_page_rules(self, domain: str) -> List[dict]:
        """List all page rules."""
        zone_id = self.get_zone_id(domain)
        return self._request("GET", f"zones/{zone_id}/pagerules").get("result", [])
    
    def create_page_rule(self, domain: str, url_pattern: str, 
                        actions: List[dict], priority: int = 1,
                        status: str = "active") -> dict:
        """
        Create a page rule.
        
        Args:
            url_pattern: URL pattern with wildcards (e.g., '*willcureton.com/api/*')
            actions: List of action dicts
            priority: Lower = higher priority
            status: active or disabled
        
        Action examples:
            [{"id": "forwarding_url", "value": {"url": "https://new.com/$1", "status_code": 301}}]
            [{"id": "always_use_https"}]
            [{"id": "cache_level", "value": "aggressive"}]
            [{"id": "browser_cache_ttl", "value": 14400}]
        """
        zone_id = self.get_zone_id(domain)
        data = {
            "targets": [{"target": "url", "constraint": {"operator": "matches", "value": url_pattern}}],
            "actions": actions,
            "priority": priority,
            "status": status
        }
        return self._request("POST", f"zones/{zone_id}/pagerules", data).get("result")
    
    def create_redirect(self, domain: str, from_pattern: str, to_url: str, 
                       status_code: int = 301) -> dict:
        """Create a redirect rule (301 or 302)."""
        return self.create_page_rule(
            domain,
            from_pattern,
            [{"id": "forwarding_url", "value": {"url": to_url, "status_code": status_code}}]
        )
    
    def delete_page_rule(self, domain: str, rule_id: str) -> dict:
        """Delete a page rule."""
        zone_id = self.get_zone_id(domain)
        return self._request("DELETE", f"zones/{zone_id}/pagerules/{rule_id}").get("result")
    
    # ===== WAF (Web Application Firewall) =====
    
    def list_waf_packages(self, domain: str) -> List[dict]:
        """List WAF packages (rulesets)."""
        zone_id = self.get_zone_id(domain)
        return self._request("GET", f"zones/{zone_id}/firewall/waf/packages").get("result", [])
    
    def get_waf_package(self, domain: str, package_id: str) -> dict:
        """Get WAF package details."""
        zone_id = self.get_zone_id(domain)
        return self._request("GET", f"zones/{zone_id}/firewall/waf/packages/{package_id}").get("result")
    
    def list_waf_rules(self, domain: str, package_id: str) -> List[dict]:
        """List rules in a WAF package."""
        zone_id = self.get_zone_id(domain)
        return self._request("GET", f"zones/{zone_id}/firewall/waf/packages/{package_id}/rules").get("result", [])
    
    def update_waf_rule(self, domain: str, package_id: str, rule_id: str, 
                       mode: str) -> dict:
        """
        Update WAF rule mode.
        mode: on, off, default, disable, simulate, block, challenge
        """
        zone_id = self.get_zone_id(domain)
        return self._request("PATCH", 
            f"zones/{zone_id}/firewall/waf/packages/{package_id}/rules/{rule_id}",
            {"mode": mode}).get("result")
    
    # ===== RATE LIMITING =====
    
    def list_rate_limits(self, domain: str) -> List[dict]:
        """List rate limiting rules."""
        zone_id = self.get_zone_id(domain)
        return self._request("GET", f"zones/{zone_id}/rate_limits").get("result", [])
    
    def create_rate_limit(self, domain: str, url_pattern: str,
                         threshold: int, period: int, action: str = "ban",
                         timeout: int = 60) -> dict:
        """
        Create a rate limiting rule.
        
        Args:
            url_pattern: URL to protect (e.g., '/api/*')
            threshold: Number of requests
            period: Time period in seconds
            action: simulate, ban, challenge, js_challenge, managed_challenge
            timeout: Ban duration in seconds
        """
        zone_id = self.get_zone_id(domain)
        data = {
            "match": {
                "request": {"url_pattern": url_pattern},
                "response": {"origin_traffic": True}
            },
            "threshold": threshold,
            "period": period,
            "action": {"mode": action, "timeout": timeout}
        }
        return self._request("POST", f"zones/{zone_id}/rate_limits", data).get("result")
    
    def delete_rate_limit(self, domain: str, rule_id: str) -> dict:
        """Delete a rate limiting rule."""
        zone_id = self.get_zone_id(domain)
        return self._request("DELETE", f"zones/{zone_id}/rate_limits/{rule_id}").get("result")
    
    # ===== SSL/TLS =====
    
    def get_ssl_setting(self, domain: str) -> str:
        """Get current SSL mode (off, flexible, full, strict)."""
        zone_id = self.get_zone_id(domain)
        return self._request("GET", f"zones/{zone_id}/settings/ssl").get("result", {}).get("value")
    
    def set_ssl_mode(self, domain: str, mode: str = "full") -> dict:
        """Set SSL mode: off, flexible, full, strict."""
        return self.update_setting(domain, "ssl", mode)
    
    def enable_always_https(self, domain: str) -> dict:
        """Force all traffic to HTTPS."""
        return self.update_setting(domain, "always_use_https", "on")
    
    def get_ssl_certificate(self, domain: str) -> dict:
        """Get SSL certificate status."""
        zone_id = self.get_zone_id(domain)
        return self._request("GET", f"zones/{zone_id}/ssl/certificate_packs").get("result", [])
    
    def enable_hsts(self, domain: str, max_age: int = 31536000, 
                   include_subdomains: bool = True, preload: bool = False) -> dict:
        """Enable HTTP Strict Transport Security."""
        zone_id = self.get_zone_id(domain)
        return self._request("PATCH", f"zones/{zone_id}/settings/security_header", {
            "value": {
                "strict_transport_security": {
                    "enabled": True,
                    "max_age": max_age,
                    "include_subdomains": include_subdomains,
                    "preload": preload,
                    "nosniff": True
                }
            }
        }).get("result")
    
    def set_min_tls_version(self, domain: str, version: str = "1.2") -> dict:
        """Set minimum TLS version (1.0, 1.1, 1.2, 1.3)."""
        return self.update_setting(domain, "min_tls_version", version)
    
    # ===== API TOKEN MANAGEMENT =====
    
    def list_tokens(self) -> List[dict]:
        """List all API tokens."""
        return self._request("GET", "user/tokens").get("result", [])
    
    def get_token_permissions(self) -> List[dict]:
        """List available permission groups."""
        return self._request("GET", "user/tokens/permission_groups").get("result", [])
    
    def update_token(self, token_id: str, name: str, policies: List[dict]) -> dict:
        """Update an API token's name and permissions."""
        return self._request("PUT", f"user/tokens/{token_id}", 
                            {"name": name, "policies": policies}).get("result")
    
    def verify_token(self) -> dict:
        """Verify current token is valid."""
        return self._request("GET", "user/tokens/verify").get("result")


# ===== HELPER FUNCTIONS =====

def quick_add(subdomain: str, domain: str = "willcureton.com", 
              ip: str = None, proxied: bool = False) -> dict:
    """
    Quickly add a subdomain to willcureton.com pointing to VPS.
    
    Example:
        quick_add("api")  # Creates api.willcureton.com -> VPS
        quick_add("cdn", proxied=True)  # With Cloudflare proxy
    """
    return cf.add_subdomain(domain, subdomain, ip, proxied)


def show_records(domain: str = "willcureton.com") -> None:
    """Display all DNS records for a domain in a readable format."""
    records = cf.list_records(domain)
    print(f"\n{'Name':<35} {'Type':<8} {'Content':<40} {'Proxy'}")
    print("-" * 95)
    for r in records:
        proxy = "🟠" if r.get("proxied") else "⚪"
        print(f"{r['name']:<35} {r['type']:<8} {r['content']:<40} {proxy}")
    print(f"\nTotal: {len(records)} records")


def show_firewall(domain: str = "willcureton.com") -> None:
    """Display firewall rules."""
    rules = cf.list_firewall_rules(domain)
    print(f"\n{'Description':<40} {'Action':<15} {'Paused'}")
    print("-" * 65)
    for r in rules:
        desc = r.get('description', r.get('filter', {}).get('expression', 'N/A'))[:38]
        print(f"{desc:<40} {r.get('action','?'):<15} {r.get('paused', False)}")
    print(f"\nTotal: {len(rules)} rules")


def show_page_rules(domain: str = "willcureton.com") -> None:
    """Display page rules."""
    rules = cf.list_page_rules(domain)
    print(f"\n{'Pattern':<50} {'Actions':<30} {'Status'}")
    print("-" * 90)
    for r in rules:
        pattern = r.get('targets', [{}])[0].get('constraint', {}).get('value', '?')[:48]
        actions = ', '.join([a.get('id', '?') for a in r.get('actions', [])])[:28]
        print(f"{pattern:<50} {actions:<30} {r.get('status', '?')}")
    print(f"\nTotal: {len(rules)} rules")


def block(ip: str, domain: str = "willcureton.com", note: str = None) -> dict:
    """Quick block an IP."""
    return cf.block_ip(domain, ip, note)


def unblock(rule_id: str, domain: str = "willcureton.com") -> dict:
    """Remove a firewall rule by ID."""
    return cf.delete_firewall_rule(domain, rule_id)


def purge(domain: str = "willcureton.com") -> dict:
    """Purge all cache for domain."""
    return cf.purge_cache(domain)


def status(domain: str = "willcureton.com") -> None:
    """Show zone status overview."""
    zone_id = cf.get_zone_id(domain)
    zone = cf.get_zone(zone_id)
    try:
        ssl = cf.get_ssl_setting(domain)
    except:
        ssl = "unknown"
    records = len(cf.list_records(domain))
    fw_rules = len(cf.list_firewall_rules(domain))
    try:
        page_rules = len(cf.list_page_rules(domain))
    except:
        page_rules = 0
    
    print(f"""
╔══════════════════════════════════════════════════════╗
║  {domain:<52} ║
╠══════════════════════════════════════════════════════╣
║  Status:      {zone.get('status', '?'):<40} ║
║  SSL Mode:    {ssl:<40} ║
║  DNS Records: {records:<40} ║
║  FW Rules:    {fw_rules:<40} ║
║  Page Rules:  {page_rules:<40} ║
║  Plan:        {zone.get('plan', {}).get('name', '?'):<40} ║
╚══════════════════════════════════════════════════════╝
""")


# Auto-instantiate for convenience
cf = CloudflareAPI()
print(f"CloudflareAPI loaded (God Mode 😈)")
print(f"Quick: quick_add(), show_records(), show_firewall(), block(), purge(), status()")
